//======================================================================================================
//
// Program created by Suresh Amberkar, Technisites Australia Pty Ltd.
//      www.technisites.com.au
//
//  Program Name: 2DMetricFasteners.cs
//
//  Version: 1.0
//
//  Date: 20th July 2016
//
//  THIS CODE COMES WITH ABSOLUTELY NO WARRANTY AND IS PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
//  WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR 
//  A PARTICULAR PURPOSE.  THE AUTHOR DOES NOT GUARANTEE THAT THE INFORMATION CONTAINED IN ASSOCIATED
//  FASTENER DATA FILES ARE ACCURATE NOR THE RESULTS PRODUCED BY IT ARE ACCURATE OR VERIFIED.
//
//======================================================================================================
//
// Program for creating 2D fasteners in NX.  Tested on NX 8, NX 9 & NX 10.
// Creates 2D Top View and Section View of
//  - Metric Socket Head Cap Screws
//  - Metric Socket Head Countersunk Screws
//  - Metric Socket Head Shoulder Screws
//
// Standard Screw Size data is stored in the following files
//  - MetricCapScrewSizeData.csv
//  - MetricCountersunkScrewSizeData.csv
//  - MetricShoulderScrewSizeData.csv
//
// Standard Screw Lengths data is stored in the following files
//  - MetricCapScrewLengths.csv
//  - MetricCountersunkScrewLengths.csv
//  - MetricShoulderScrewLengths.csv
//
// The 'Screw Size' list in the program is populated by reading the appropriate
//  Standard Screw Size data file.
//
// Appropriate screw length is selected by measuring the distance between the
//  - Base Line (the line from where the hole is tapped)
//  - Boundary Line (the line from where the screw is inserted)
//  plus the thread engagement length, minus the counter bore depth. Counter bore depth
//  is adjusted to accommodate standard screw length.
//
// Shoulder screws are drawn with a clearance on body diameter and 2mm clearance under
//  the head.
//
// Top view is always drawn in the XC YC plane.
// 
// Section view requires three lines to be picked.
//  - The screw center line
//  - The base line (where the tap hole begins)
//  - The boundary line (where the screw is inserted from)
//  - The base line and boundary line are expected to be parallel to each other and 
//  perpendicular to the center line. All three lines must be on the same plane.
//  They can be in any orientation as long as the above conditions are met.
//
// Program expects the associated screw data csv files and bitmap files for the dialog to be
//  in a directory called '2dMetricFasteners'.  This directory should be placed under
//  directory pointed to by 'UGII_SITE_DIR' environment variable.
//
//======================================================================================================


using System;
using System.IO;
using System.Collections.Generic;
using System.Windows.Forms;
using System.Drawing;
using NXOpen;
using NXOpen.UF;

//[Microsoft.VisualBasic.CompilerServices.DesignerGenerated()]
partial class Form3 : System.Windows.Forms.Form
{
    private string screwCategoryText;
    private string viewOptionText;
    private string fontOptionText;
    private string[] screwSizes;
    private string[] shoulderScrewSizes;
    private string theAppDirPath;
    //Form overrides dispose to clean up the component list.
    internal System.Windows.Forms.Button Button1;
    internal System.Windows.Forms.Button Button2;
    private GroupBox groupBox1;
    private RadioButton radioButton_TappedHole;
    private RadioButton radioButton_ShoulderBolt;
    private RadioButton radioButton_CSK_Screw;
    private RadioButton radioButton_CapScrew;
    private ComboBox comboBox1;
    private Label label1;
    private GroupBox groupBox2;
    private RadioButton radioButton_SectionView;
    private RadioButton radioButton_TopView;
    private GroupBox groupBox3;
    private RadioButton radioButton_FontHidden;
    private RadioButton radioButton_FontSolid;
    private PictureBox pictureBox1;
    //Required by the Windows Form Designer

    private System.ComponentModel.IContainer components;
    //NOTE: The following procedure is required by the Windows Form Designer
    //It can be modified using the Windows Form Designer.  
    //Do not modify it using the code editor.
    private void InitializeComponent()
    {
            this.Button1 = new System.Windows.Forms.Button();
            this.Button2 = new System.Windows.Forms.Button();
            this.groupBox1 = new System.Windows.Forms.GroupBox();
            this.radioButton_TappedHole = new System.Windows.Forms.RadioButton();
            this.radioButton_ShoulderBolt = new System.Windows.Forms.RadioButton();
            this.radioButton_CSK_Screw = new System.Windows.Forms.RadioButton();
            this.radioButton_CapScrew = new System.Windows.Forms.RadioButton();
            this.comboBox1 = new System.Windows.Forms.ComboBox();
            this.label1 = new System.Windows.Forms.Label();
            this.groupBox2 = new System.Windows.Forms.GroupBox();
            this.radioButton_SectionView = new System.Windows.Forms.RadioButton();
            this.radioButton_TopView = new System.Windows.Forms.RadioButton();
            this.groupBox3 = new System.Windows.Forms.GroupBox();
            this.radioButton_FontHidden = new System.Windows.Forms.RadioButton();
            this.radioButton_FontSolid = new System.Windows.Forms.RadioButton();
            this.pictureBox1 = new System.Windows.Forms.PictureBox();
            this.groupBox1.SuspendLayout();
            this.groupBox2.SuspendLayout();
            this.groupBox3.SuspendLayout();
            ((System.ComponentModel.ISupportInitialize)(this.pictureBox1)).BeginInit();
            this.SuspendLayout();
            // 
            // Button1
            // 
            this.Button1.DialogResult = System.Windows.Forms.DialogResult.Cancel;
            this.Button1.Font = new System.Drawing.Font("Microsoft Sans Serif", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((byte)(0)));
            this.Button1.Location = new System.Drawing.Point(125, 558);
            this.Button1.Margin = new System.Windows.Forms.Padding(2);
            this.Button1.Name = "Button1";
            this.Button1.Size = new System.Drawing.Size(81, 23);
            this.Button1.TabIndex = 0;
            this.Button1.Text = "Cancel";
            this.Button1.UseVisualStyleBackColor = true;
            // 
            // Button2
            // 
            this.Button2.DialogResult = System.Windows.Forms.DialogResult.OK;
            this.Button2.Location = new System.Drawing.Point(43, 557);
            this.Button2.Name = "Button2";
            this.Button2.Size = new System.Drawing.Size(75, 23);
            this.Button2.TabIndex = 15;
            this.Button2.Text = "OK";
            this.Button2.UseVisualStyleBackColor = true;
            this.Button2.Click += new System.EventHandler(this.Button2_Click);
            // 
            // groupBox1
            // 
            this.groupBox1.Controls.Add(this.radioButton_TappedHole);
            this.groupBox1.Controls.Add(this.radioButton_ShoulderBolt);
            this.groupBox1.Controls.Add(this.radioButton_CSK_Screw);
            this.groupBox1.Controls.Add(this.radioButton_CapScrew);
            this.groupBox1.Location = new System.Drawing.Point(13, 13);
            this.groupBox1.Name = "groupBox1";
            this.groupBox1.Size = new System.Drawing.Size(224, 125);
            this.groupBox1.TabIndex = 16;
            this.groupBox1.TabStop = false;
            this.groupBox1.Text = "Category";
            // 
            // radioButton_TappedHole
            // 
            this.radioButton_TappedHole.AutoSize = true;
            this.radioButton_TappedHole.Location = new System.Drawing.Point(7, 92);
            this.radioButton_TappedHole.Name = "radioButton_TappedHole";
            this.radioButton_TappedHole.Size = new System.Drawing.Size(87, 17);
            this.radioButton_TappedHole.TabIndex = 3;
            this.radioButton_TappedHole.TabStop = true;
            this.radioButton_TappedHole.Text = "Tapped Hole";
            this.radioButton_TappedHole.UseVisualStyleBackColor = true;
            this.radioButton_TappedHole.CheckedChanged += new System.EventHandler(this.radioButton_CapScrew_CheckedChanged);
            // 
            // radioButton_ShoulderBolt
            // 
            this.radioButton_ShoulderBolt.AutoSize = true;
            this.radioButton_ShoulderBolt.Location = new System.Drawing.Point(7, 68);
            this.radioButton_ShoulderBolt.Name = "radioButton_ShoulderBolt";
            this.radioButton_ShoulderBolt.Size = new System.Drawing.Size(88, 17);
            this.radioButton_ShoulderBolt.TabIndex = 2;
            this.radioButton_ShoulderBolt.TabStop = true;
            this.radioButton_ShoulderBolt.Text = "Shoulder Bolt";
            this.radioButton_ShoulderBolt.UseVisualStyleBackColor = true;
            this.radioButton_ShoulderBolt.CheckedChanged += new System.EventHandler(this.radioButton_CapScrew_CheckedChanged);
            // 
            // radioButton_CSK_Screw
            // 
            this.radioButton_CSK_Screw.AutoSize = true;
            this.radioButton_CSK_Screw.Location = new System.Drawing.Point(7, 44);
            this.radioButton_CSK_Screw.Name = "radioButton_CSK_Screw";
            this.radioButton_CSK_Screw.Size = new System.Drawing.Size(118, 17);
            this.radioButton_CSK_Screw.TabIndex = 1;
            this.radioButton_CSK_Screw.TabStop = true;
            this.radioButton_CSK_Screw.Text = "Countersunk Screw";
            this.radioButton_CSK_Screw.UseVisualStyleBackColor = true;
            this.radioButton_CSK_Screw.CheckedChanged += new System.EventHandler(this.radioButton_CapScrew_CheckedChanged);
            // 
            // radioButton_CapScrew
            // 
            this.radioButton_CapScrew.AutoSize = true;
            this.radioButton_CapScrew.Location = new System.Drawing.Point(7, 20);
            this.radioButton_CapScrew.Name = "radioButton_CapScrew";
            this.radioButton_CapScrew.Size = new System.Drawing.Size(77, 17);
            this.radioButton_CapScrew.TabIndex = 0;
            this.radioButton_CapScrew.TabStop = true;
            this.radioButton_CapScrew.Text = "Cap Screw";
            this.radioButton_CapScrew.UseVisualStyleBackColor = true;
            this.radioButton_CapScrew.CheckedChanged += new System.EventHandler(this.radioButton_CapScrew_CheckedChanged);
            // 
            // comboBox1
            // 
            this.comboBox1.FormattingEnabled = true;
            this.comboBox1.Location = new System.Drawing.Point(115, 144);
            this.comboBox1.Name = "comboBox1";
            this.comboBox1.Size = new System.Drawing.Size(121, 21);
            this.comboBox1.TabIndex = 17;
            // 
            // label1
            // 
            this.label1.AutoSize = true;
            this.label1.Location = new System.Drawing.Point(17, 147);
            this.label1.Name = "label1";
            this.label1.Size = new System.Drawing.Size(27, 13);
            this.label1.TabIndex = 18;
            this.label1.Text = "Size";
            // 
            // groupBox2
            // 
            this.groupBox2.Controls.Add(this.radioButton_SectionView);
            this.groupBox2.Controls.Add(this.radioButton_TopView);
            this.groupBox2.Location = new System.Drawing.Point(13, 177);
            this.groupBox2.Name = "groupBox2";
            this.groupBox2.Size = new System.Drawing.Size(224, 58);
            this.groupBox2.TabIndex = 19;
            this.groupBox2.TabStop = false;
            this.groupBox2.Text = "View";
            // 
            // radioButton_SectionView
            // 
            this.radioButton_SectionView.AutoSize = true;
            this.radioButton_SectionView.Location = new System.Drawing.Point(133, 19);
            this.radioButton_SectionView.Name = "radioButton_SectionView";
            this.radioButton_SectionView.Size = new System.Drawing.Size(87, 17);
            this.radioButton_SectionView.TabIndex = 1;
            this.radioButton_SectionView.TabStop = true;
            this.radioButton_SectionView.Text = "Section View";
            this.radioButton_SectionView.UseVisualStyleBackColor = true;
            this.radioButton_SectionView.CheckedChanged += new System.EventHandler(this.radioButton_FontSolid_CheckedChanged);
            // 
            // radioButton_TopView
            // 
            this.radioButton_TopView.AutoSize = true;
            this.radioButton_TopView.Location = new System.Drawing.Point(7, 20);
            this.radioButton_TopView.Name = "radioButton_TopView";
            this.radioButton_TopView.Size = new System.Drawing.Size(70, 17);
            this.radioButton_TopView.TabIndex = 0;
            this.radioButton_TopView.TabStop = true;
            this.radioButton_TopView.Text = "Top View";
            this.radioButton_TopView.UseVisualStyleBackColor = true;
            this.radioButton_TopView.CheckedChanged += new System.EventHandler(this.radioButton_FontSolid_CheckedChanged);
            // 
            // groupBox3
            // 
            this.groupBox3.Controls.Add(this.radioButton_FontHidden);
            this.groupBox3.Controls.Add(this.radioButton_FontSolid);
            this.groupBox3.Location = new System.Drawing.Point(13, 242);
            this.groupBox3.Name = "groupBox3";
            this.groupBox3.Size = new System.Drawing.Size(224, 66);
            this.groupBox3.TabIndex = 20;
            this.groupBox3.TabStop = false;
            this.groupBox3.Text = "Font";
            // 
            // radioButton_FontHidden
            // 
            this.radioButton_FontHidden.AutoSize = true;
            this.radioButton_FontHidden.Location = new System.Drawing.Point(133, 29);
            this.radioButton_FontHidden.Name = "radioButton_FontHidden";
            this.radioButton_FontHidden.Size = new System.Drawing.Size(59, 17);
            this.radioButton_FontHidden.TabIndex = 1;
            this.radioButton_FontHidden.TabStop = true;
            this.radioButton_FontHidden.Text = "Hidden";
            this.radioButton_FontHidden.UseVisualStyleBackColor = true;
            // 
            // radioButton_FontSolid
            // 
            this.radioButton_FontSolid.AutoSize = true;
            this.radioButton_FontSolid.Location = new System.Drawing.Point(7, 29);
            this.radioButton_FontSolid.Name = "radioButton_FontSolid";
            this.radioButton_FontSolid.Size = new System.Drawing.Size(48, 17);
            this.radioButton_FontSolid.TabIndex = 0;
            this.radioButton_FontSolid.TabStop = true;
            this.radioButton_FontSolid.Text = "Solid";
            this.radioButton_FontSolid.UseVisualStyleBackColor = true;
            this.radioButton_FontSolid.CheckedChanged += new System.EventHandler(this.radioButton_FontSolid_CheckedChanged);
            // 
            // pictureBox1
            // 
            this.pictureBox1.InitialImage = null;
            this.pictureBox1.Location = new System.Drawing.Point(12, 318);
            this.pictureBox1.Name = "pictureBox1";
            this.pictureBox1.Size = new System.Drawing.Size(224, 224);
            this.pictureBox1.SizeMode = System.Windows.Forms.PictureBoxSizeMode.AutoSize;
            this.pictureBox1.TabIndex = 21;
            this.pictureBox1.TabStop = false;
            // 
            // Form3
            // 
            this.AutoScaleDimensions = new System.Drawing.SizeF(6F, 13F);
            this.AutoScaleMode = System.Windows.Forms.AutoScaleMode.Font;
            this.AutoSize = true;
            this.ClientSize = new System.Drawing.Size(249, 592);
            this.Controls.Add(this.pictureBox1);
            this.Controls.Add(this.groupBox3);
            this.Controls.Add(this.groupBox2);
            this.Controls.Add(this.label1);
            this.Controls.Add(this.comboBox1);
            this.Controls.Add(this.groupBox1);
            this.Controls.Add(this.Button2);
            this.Controls.Add(this.Button1);
            this.Margin = new System.Windows.Forms.Padding(2);
            this.Name = "Form3";
            this.Text = "Attributes";
            this.groupBox1.ResumeLayout(false);
            this.groupBox1.PerformLayout();
            this.groupBox2.ResumeLayout(false);
            this.groupBox2.PerformLayout();
            this.groupBox3.ResumeLayout(false);
            this.groupBox3.PerformLayout();
            ((System.ComponentModel.ISupportInitialize)(this.pictureBox1)).EndInit();
            this.ResumeLayout(false);
            this.PerformLayout();

    }


    private void radioButton_CapScrew_CheckedChanged(object sender, EventArgs e)
    {

        foreach (Control control in this.groupBox1.Controls)
        {
            if (control is RadioButton)
            {
                RadioButton radio = control as RadioButton;
                if (radio.Checked)
                {
                    screwCategoryText = radio.Text;
                    if (screwCategoryText == "Shoulder Bolt")
                    {
                        fillComboItems(shoulderScrewSizes);

                    }
                    else
                    {
                        fillComboItems(screwSizes);

                    }
                    setPictureBoxImage();
                }
            }
        }

    }

    private void fillComboItems(string[] theItems)
    {
        comboBox1.Items.Clear();

        for (int i = 0; i < theItems.Length; i++ )
        {
            comboBox1.Items.Add(theItems[i].ToString());
        }
        comboBox1.SelectedIndex = 0;
    }

    private void Button2_Click(object sender, EventArgs e)
    {

    }

    private void radioButton_FontSolid_CheckedChanged(object sender, EventArgs e)
    {
        foreach (Control control in this.groupBox2.Controls)
        {
            if (control is RadioButton)
            {
                RadioButton radio = control as RadioButton;
                if (radio.Checked)
                {
                    viewOptionText = radio.Text;
                }
            }
        }

        foreach (Control control in this.groupBox3.Controls)
        {
            if (control is RadioButton)
            {
                RadioButton radio = control as RadioButton;
                if (radio.Checked)
                {
                    fontOptionText = radio.Text;
                }
            }
        }
        setPictureBoxImage();
    }

    private void setPictureBoxImage()
    {
        string imageFile = theAppDirPath + "Bitmaps//";
        imageFile += screwCategoryText + " " + viewOptionText + " " + fontOptionText + ".png";
        pictureBox1.ImageLocation = imageFile;
    }



    public static void Main()
    {
        ScrewClass theScrewClass = null;

        try
        {
            Form3 theForm = new Form3();
            theScrewClass = new ScrewClass();

            if (theScrewClass.theAppDirPath.Length == 0)
            {
                MessageBox.Show("Error reading environment variable UGII_SITE_DIR",
                    "Environment Variable", MessageBoxButtons.OK);
                return;
            }

            theScrewClass.theAppDirPath = theScrewClass.theAppDirPath + "\\2dMetricFasteners\\";


            if (!Directory.Exists(theScrewClass.theAppDirPath))
            {
                MessageBox.Show("Error finding directory "+ theScrewClass.theAppDirPath,
                    "App Directory", MessageBoxButtons.OK);
                return;
            }

            theForm.theAppDirPath = theScrewClass.theAppDirPath;

            string fileNameShoulderScrew = theScrewClass.theAppDirPath + "MetricShoulderScrewSizeData.csv";
            string fileNameCapScrew = theScrewClass.theAppDirPath + "MetricCapScrewSizeData.csv";
            string fileNameCSKScrew = theScrewClass.theAppDirPath + "MetricCountersunkScrewSizeData.csv";

            theForm.screwSizes = theScrewClass.getScrewSizes(fileNameCapScrew);
            theForm.shoulderScrewSizes = theScrewClass.getScrewSizes(fileNameShoulderScrew);

            theForm.InitializeComponent();
            theForm.fillComboItems(theForm.screwSizes);
            theForm.radioButton_CapScrew.Checked = true;
            theForm.radioButton_FontSolid.Checked = true;
            theForm.radioButton_TopView.Checked = true;
            theForm.Text = "Metric Screws 2D";

            UFSession theUFS = UFSession.GetUFSession();
            theUFS.Ui.SetPrompt("Choose Screw Options and Press OK");

            if (theForm.ShowDialog() == System.Windows.Forms.DialogResult.OK)
            {
                theScrewClass.screwCategory = theForm.screwCategoryText;
                theScrewClass.viewOption = theForm.viewOptionText;
                theScrewClass.fontOption = theForm.fontOptionText;

                theScrewClass.screwSize = theForm.comboBox1.SelectedItem.ToString();

                Point3d originPoint = new Point3d(0,0,0);

                if (theScrewClass.screwCategory == "Cap Screw" ||
                    theScrewClass.screwCategory == "Tapped Hole")
                {
                    theScrewClass.getScrewData(fileNameCapScrew);

                    if (theScrewClass.viewOption == "Top View")
                    {
                        while (theScrewClass.speicfyPointForTopView(ref originPoint))
                        {
                            if (theScrewClass.screwCategory == "Cap Screw")
                                theScrewClass.drawTopViewCapScrew(originPoint);
                            else
                                theScrewClass.drawTopViewTappedHole(originPoint);
                        }
                    }
                    else
                    {
                        if (theScrewClass.getSelection())
                        {
                            if (theScrewClass.screwCategory == "Cap Screw")
                                theScrewClass.drawSectionViewCapScrew();
                            else
                                theScrewClass.drawSectionViewTappedHole();
                        }

                    }
                }
                else if (theScrewClass.screwCategory == "Countersunk Screw")
                {
                    theScrewClass.getScrewData(fileNameCSKScrew);

                    if (theScrewClass.viewOption == "Top View")
                    {
                        while (theScrewClass.speicfyPointForTopView(ref originPoint))
                        {
                            theScrewClass.drawTopViewCountersunkScrew(originPoint);
                        }
                    }
                    else
                    {
                        if (theScrewClass.getSelection())
                        {
                            theScrewClass.drawSectionViewCSKScrew();
                        }

                    }
                }
                else if (theScrewClass.screwCategory == "Shoulder Bolt")
                {
                    theScrewClass.getScrewData(fileNameShoulderScrew);

                    if (theScrewClass.viewOption == "Top View")
                    {
                        while (theScrewClass.speicfyPointForTopView(ref originPoint))
                        {
                            theScrewClass.drawTopViewShoulderScrew(originPoint);
                        }
                    }
                    else
                    {
                        if (theScrewClass.getSelection())
                        {
                            theScrewClass.drawSectionViewShoulderScrew();
                        }

                    }
                }

            }

        }
        catch (Exception Ex)
        {
            MessageBox.Show( Ex.Message, "Metric Screws 2D");
        }

    }

}


public class ScrewClass
{

    private static Session s;
    private static UFSession ufs;
    private static UI theUI;
    private static ListingWindow lw;

    private Part wp;
    private Part dp;

    private string screwCategoryString;
    private string viewOptionString;
    private string fontOptionString;
    private string screwSizeString;
    public string theAppDirPath;

    private double d_nom_size,
        d_tap_size,
        d_tap_drill_dia,
        d_tap_drill_depth,
        d_thread_len,
        d_tap_depth,
        d_clr_dia,
        d_cbore_dia,
        d_cbore_depth,
        d_csk_dia,
        d_head_dia,
        d_head_height,
        d_hex_size,
        d_csk_head_flat,
        d_csk_head_angle,
        d_shoulder_screw_depth_clearance;

    private double base_to_bnd_dist;
    private double screw_len;

    private Line centerLine;
    private Line baseLine;
    private Line boundLine;

    private NXOpen.Point refPoint;

    private Vector3d dir_X;
    private Vector3d dir_Y;

    private List<double> std_screw_lens;


    public ScrewClass()
    {

        try
        {
            s = Session.GetSession();
            ufs = UFSession.GetUFSession();
            theUI = UI.GetUI();
            lw = s.ListingWindow;
            wp = s.Parts.Work;
            dp = s.Parts.Display;

            screwCategory = "Cap Screw";
            viewOption = "Top View";
            fontOption = "Solid";

            std_screw_lens = new List<double>();

            ufs.UF.TranslateVariable("UGII_SITE_DIR", out theAppDirPath);

            d_shoulder_screw_depth_clearance = 2.0;

        }
        catch (Exception ex)
        {
            throw ex;
        }

    }

    public string screwCategory
    {
        get { return screwCategoryString; }
        set { screwCategoryString = value; }
    }

    public string viewOption
    {
        get { return viewOptionString; }
        set { viewOptionString = value; }
    }

    public string fontOption
    {
        get { return fontOptionString; }
        set { fontOptionString = value; }
    }

    public string screwSize
    {
        get { return screwSizeString; }
        set { screwSizeString = value; }
    }


    //Get screw sizes to display in combo box from csv file
    public string[] getScrewSizes(string path)
    {
        List<string> parsedData = new List<string>();
        //lw.Open();
        //lw.WriteLine(path);

        try
        {
            using (StreamReader readFile = new StreamReader(path))
            {
                string line;
                string[] row;

                int i = 0;
                while ((line = readFile.ReadLine()) != null)
                {
                    if (i > 0)
                    {
                        row = line.Split(',');
                        if (path.Contains("Shoulder"))
                            parsedData.Add("Dia " + row[0]);
                        else
                            parsedData.Add("M" + row[0]);

                        //lw.WriteLine(row[0]);
                    }
                    i++;

                }
            }
        }
        catch (Exception e)
        {
            MessageBox.Show(e.Message);
        }

        return parsedData.ToArray();
    }

    //Get screw data of the selected type and size from csv file
    public void getScrewData(string path)
    {
        try
        {
            using (StreamReader readFile = new StreamReader(path))
            {
                string line;
                string[] row;

                int i = 0;
                while ((line = readFile.ReadLine()) != null)
                {
                    if (i > 0)
                    {
                        row = line.Split(',');
                        if (screwCategory == "Cap Screw" || screwCategory == "Tapped Hole")
                        {
                            if (row.Length != 9)
                            {
                                MessageBox.Show("Array length not of expected length." + 
                                "Reading Cap Screw Table!!");
                                return;
                            }
                            if ("M" + row[0] == screwSize)
                            {
                                d_nom_size = Convert.ToDouble(row[0]);
                                d_tap_size = d_nom_size;
                                d_tap_drill_dia = Convert.ToDouble(row[1]);
                                d_tap_drill_depth = Convert.ToDouble(row[2]);
                                d_tap_depth = Convert.ToDouble(row[3]);
                                d_clr_dia = Convert.ToDouble(row[4]);
                                d_cbore_dia = Convert.ToDouble(row[5]);
                                d_cbore_depth = Convert.ToDouble(row[6]);
                                d_head_dia = Convert.ToDouble(row[7]);
                                d_hex_size = Convert.ToDouble(row[8]);
                            }
                        }
                        else if (screwCategory == "Countersunk Screw")
                        {
                            if (row.Length != 10)
                            {
                                MessageBox.Show("Array length not of expected length." +
                                "Reading Countersunk Screw Table!!");
                                return;
                            }
                            if ("M" + row[0] == screwSize)
                            {
                                d_nom_size = Convert.ToDouble(row[0]);
                                d_tap_size = d_nom_size;
                                d_tap_drill_dia = Convert.ToDouble(row[1]);
                                d_tap_drill_depth = Convert.ToDouble(row[2]);
                                d_tap_depth = Convert.ToDouble(row[3]);
                                d_clr_dia = Convert.ToDouble(row[4]);
                                d_csk_dia = Convert.ToDouble(row[5]);
                                d_head_dia = Convert.ToDouble(row[6]);
                                d_hex_size = Convert.ToDouble(row[7]);
                                d_csk_head_angle = Convert.ToDouble(row[8]);
                                d_csk_head_flat = Convert.ToDouble(row[9]);
                            }
                        }
                        else if (screwCategory == "Shoulder Bolt")
                        {
                            if (row.Length != 12)
                            {
                                MessageBox.Show("Array length not of expected length." +
                                "Reading Shoulder Screw Table!!");
                                return;
                            }
                            if ("Dia " + row[0] == screwSize)
                            {
                                d_nom_size = Convert.ToDouble(row[0]);
                                d_tap_size = Convert.ToDouble(row[1]);
                                d_tap_drill_dia = Convert.ToDouble(row[2]);
                                d_tap_drill_depth = Convert.ToDouble(row[3]);
                                d_tap_depth = Convert.ToDouble(row[4]);
                                d_clr_dia = Convert.ToDouble(row[5]);
                                d_cbore_dia = Convert.ToDouble(row[6]);
                                d_cbore_depth = Convert.ToDouble(row[7]);
                                d_head_dia = Convert.ToDouble(row[8]);
                                d_head_height = Convert.ToDouble(row[9]);
                                d_hex_size = Convert.ToDouble(row[10]);
                                d_thread_len = Convert.ToDouble(row[11]);
                            }
                        }

                    }
                    i++;

                }
            }
        }
        catch (Exception e)
        {
            MessageBox.Show(e.Message);
        }

        return;
    }


    //Get screw data of the selected type and size from csv file
    private void getStdScrewLengths()
    {
        string fileName;

        if (screwCategory == "Cap Screw")
            fileName = "MetricCapScrewLengths.csv";
        else if (screwCategory == "Countersunk Screw")
            fileName = "MetricCountersunkScrewLengths.csv";
        else
            fileName = "MetricShoulderScrewLengths.csv";

        fileName = theAppDirPath + fileName;

        if (!File.Exists(fileName))
        {
            MessageBox.Show("Error finding file " + fileName,
                "Screw Lengths File", MessageBoxButtons.OK);
            return;
        }

        try
        {
            using (StreamReader readFile = new StreamReader(fileName))
            {
                string line;
                string[] row;

                int i = 0;
                while ((line = readFile.ReadLine()) != null)
                {
                    if (i > 0)
                    {
                        row = line.Split(',');
                        string theSize;
                        if (screwCategory == "Shoulder Bolt")
                            theSize = "Dia " + row[0];
                        else
                            theSize = row[0];

                        if (row.Length != 2)
                        {
                            MessageBox.Show("Array length not of expected length." +
                            "Reading Screw Lengths!!");
                            return;
                        }
                        if (theSize == screwSize)
                        {
                            std_screw_lens.Add(Convert.ToDouble(row[1]));
                        }

                    }
                    i++;

                }
            }
        }
        catch (Exception e)
        {
            MessageBox.Show(e.Message);
        }

        return;
    }

    //Specify point
    public bool speicfyPointForTopView(ref Point3d thePoint)
    {
        int[] mode = { 0, 0 };
        double[] point = { 0, 0, 0 };
        int ret = 0;

        ufs.Ui.LockUgAccess(UFConstants.UF_UI_FROM_CUSTOM);
            
        ret = ufs.Ui.PointSubfunction("specify point", mode, 0, point);

        ufs.Ui.UnlockUgAccess(UFConstants.UF_UI_FROM_CUSTOM);

        if (ret == 5)
        {
            thePoint.X = point[0];
            thePoint.Y = point[1];
            thePoint.Z = point[2];

            return true;
        }

        return false;
    }

    //Draw top view of Cap Screw
    public void drawTopViewCapScrew(Point3d originPoint)
    {
        Vector3d XC;
        Vector3d YC;
        List<TaggedObject> theCurves = new List<TaggedObject>();

        dp.WCS.CoordinateSystem.GetDirections(out XC, out YC);

        Session.UndoMarkId markId1 = s.SetUndoMark(Session.MarkVisibility.Visible, screwSizeString + " " + viewOptionString);

        //c'bore dia
        Arc theArcCbDia = wp.Curves.CreateArc(originPoint, XC, YC, d_cbore_dia / 2, 0, 2 * Math.PI);
        if (fontOption == "Hidden")
            setCurveFontWidthHidden(theArcCbDia);
        theCurves.Add(theArcCbDia);

        //screw head dia
        Arc theArcHeadDia = wp.Curves.CreateArc(originPoint, XC, YC, d_head_dia / 2, 0, 2 * Math.PI);
        if (fontOption == "Hidden")
            setCurveFontWidthHidden(theArcHeadDia);
        theCurves.Add(theArcHeadDia);

        //screw nominal dia
        Arc theArcNomDia = wp.Curves.CreateArc(originPoint, XC, YC, d_nom_size / 2, 0.4 * Math.PI, 2 * Math.PI);
        setCurveFontWidthHidden(theArcNomDia);
        theCurves.Add(theArcNomDia);

        //screw tap drill dia
        Arc theArcTapDrill = wp.Curves.CreateArc(originPoint, XC, YC, d_tap_drill_dia / 2, 0, 2 * Math.PI);
        setCurveFontWidthHidden(theArcTapDrill);
        theCurves.Add(theArcTapDrill);

        //Create Hexagon if font is solid
        if (fontOption == "Solid")
        {
            List<TaggedObject> hexLines = new List<TaggedObject>();

            hexLines = createHexagon(d_hex_size / 2, originPoint);

            theCurves.AddRange(hexLines);
        }

        //Create group
        createtheGroup(theCurves);
    }

    //Create hexagon
    private List<TaggedObject> createHexagon(double inscribedRad, Point3d originPoint)
    {
        List<TaggedObject> hexLines = new List<TaggedObject>();
        double halfSideLen;
        double cornerLen;

        Vector3d XC;
        Vector3d YC;

        dp.WCS.CoordinateSystem.GetDirections(out XC, out YC);

        halfSideLen = Math.Tan (30 * Math.PI/180) * inscribedRad;
	    cornerLen =  inscribedRad / Math.Cos (30 * Math.PI/180);

        Point3d p0 = offset_point(originPoint, XC, -halfSideLen);
        Point3d p1 = offset_point(p0, YC, -inscribedRad);
        Point3d p2 = offset_point(p1, XC, halfSideLen * 2);
        Point3d p3 = offset_point(originPoint, XC, cornerLen);
        Point3d p4 = offset_point(p2, YC, inscribedRad * 2);
        Point3d p5 = offset_point(p4, XC, -halfSideLen * 2);
        Point3d p6 = offset_point(originPoint, XC, -cornerLen);


        NXObject line1 = wp.Curves.CreateLine(p1, p2);
        hexLines.Add(line1);
        NXObject line2 = wp.Curves.CreateLine(p2, p3);
        hexLines.Add(line2);
        NXObject line3 = wp.Curves.CreateLine(p3, p4);
        hexLines.Add(line3);
        NXObject line4 = wp.Curves.CreateLine(p4, p5);
        hexLines.Add(line4);
        NXObject line5 = wp.Curves.CreateLine(p5, p6);
        hexLines.Add(line5);
        NXObject line6 = wp.Curves.CreateLine(p6, p1);
        hexLines.Add(line6);

        return hexLines;
    }

    private void setCurveFontWidthHidden(DisplayableObject theObj)
    {
        theObj.LineFont = DisplayableObject.ObjectFont.Dashed;
        theObj.LineWidth = DisplayableObject.ObjectWidth.Thin;
    }


    //Draw tapped hole top view
    public void drawTopViewTappedHole(Point3d originPoint)
    {
        Vector3d XC;
        Vector3d YC;
        List<TaggedObject> theCurves = new List<TaggedObject>();

        dp.WCS.CoordinateSystem.GetDirections(out XC, out YC);

        Session.UndoMarkId markId1 = s.SetUndoMark(Session.MarkVisibility.Visible, screwSizeString + " " + viewOptionString);

        //screw nominal dia
        Arc theArcNomDia = wp.Curves.CreateArc(originPoint, XC, YC, d_tap_size / 2, 0.4 * Math.PI, 2 * Math.PI);
        if ( fontOption == "Hidden")
            setCurveFontWidthHidden(theArcNomDia);
        theCurves.Add(theArcNomDia);

        //screw tap drill dia
        Arc theArcTapDrill = wp.Curves.CreateArc(originPoint, XC, YC, d_tap_drill_dia / 2, 0, 2 * Math.PI);
        if (fontOption == "Hidden")
            setCurveFontWidthHidden(theArcTapDrill);
        theCurves.Add(theArcTapDrill);

        createtheGroup(theCurves);
    }


    //Draw Countersunk hole top view
    public void drawTopViewCountersunkScrew(Point3d originPoint)
    {
        Vector3d XC;
        Vector3d YC;
        List<TaggedObject> theCurves = new List<TaggedObject>();

        dp.WCS.CoordinateSystem.GetDirections(out XC, out YC);

        Session.UndoMarkId markId1 = s.SetUndoMark(Session.MarkVisibility.Visible, screwSizeString + " " + viewOptionString);

        //CSK dia
        Arc theArcCskDia = wp.Curves.CreateArc(originPoint, XC, YC, d_csk_dia / 2, 0, 2 * Math.PI);
        if (fontOption == "Hidden")
            setCurveFontWidthHidden(theArcCskDia);
        theCurves.Add(theArcCskDia);

        //Screw head dia
        Arc theArcHeadDia = wp.Curves.CreateArc(originPoint, XC, YC, d_head_dia / 2, 0, 2 * Math.PI);
        if (fontOption == "Hidden")
            setCurveFontWidthHidden(theArcHeadDia);
        theCurves.Add(theArcHeadDia);

        //screw nominal dia
        Arc theArcNomDia = wp.Curves.CreateArc(originPoint, XC, YC, d_tap_size / 2, 0.4 * Math.PI, 2 * Math.PI);
        setCurveFontWidthHidden(theArcNomDia);
        theCurves.Add(theArcNomDia);

        //screw tapdrill dia
        Arc theArcTapDrill = wp.Curves.CreateArc(originPoint, XC, YC, d_tap_drill_dia / 2, 0, 2 * Math.PI);
        setCurveFontWidthHidden(theArcTapDrill);
        theCurves.Add(theArcTapDrill);

        List<TaggedObject> hexLines = new List<TaggedObject>();

        hexLines = createHexagon(d_hex_size / 2, originPoint);

        if (fontOption == "Hidden")
        {
            foreach (DisplayableObject theObj in hexLines)
            {
                theObj.LineFont = DisplayableObject.ObjectFont.Dashed;
            }
        }
            
        theCurves.AddRange(hexLines);

        createtheGroup(theCurves);

    }


    public void drawTopViewShoulderScrew(Point3d originPoint)
    {
        Vector3d XC;
        Vector3d YC;
        List<TaggedObject> theCurves = new List<TaggedObject>();

        dp.WCS.CoordinateSystem.GetDirections(out XC, out YC);

        Session.UndoMarkId markId1 = s.SetUndoMark(Session.MarkVisibility.Visible, screwSizeString + " " + viewOptionString);

        //c'bore dia
        Arc theArcCbDia = wp.Curves.CreateArc(originPoint, XC, YC, d_cbore_dia / 2, 0, 2 * Math.PI);

        if (fontOption == "Hidden")
            setCurveFontWidthHidden(theArcCbDia);

        theCurves.Add(theArcCbDia);

        //screw head dia
        Arc theArcHeadDia = wp.Curves.CreateArc(originPoint, XC, YC, d_head_dia / 2, 0, 2 * Math.PI);

        if (fontOption == "Hidden")
            setCurveFontWidthHidden(theArcHeadDia);

        theCurves.Add(theArcHeadDia);

        //screw nominal dia
        Arc theArc = wp.Curves.CreateArc(originPoint, XC, YC, d_nom_size / 2, 0, 2 * Math.PI);
        setCurveFontWidthHidden(theArc);

        theCurves.Add(theArc);

        //screw tap dia
        Arc theTapArc = wp.Curves.CreateArc(originPoint, XC, YC, d_tap_size / 2, 0.4 * Math.PI, 2 * Math.PI);
        setCurveFontWidthHidden(theTapArc);

        theCurves.Add(theTapArc);

        //screw tap drill dia
        Arc theArcTapDrill = wp.Curves.CreateArc(originPoint, XC, YC, d_tap_drill_dia / 2, 0, 2 * Math.PI);
        setCurveFontWidthHidden(theArcTapDrill);

        theCurves.Add(theArcTapDrill);

        List<TaggedObject> hexLines = new List<TaggedObject>();

        hexLines = createHexagon(d_hex_size / 2, originPoint);

        if (fontOption == "Hidden")
        {
            foreach (DisplayableObject theObj in hexLines)
            {
                theObj.LineFont = DisplayableObject.ObjectFont.Dashed;
            }
        }

        theCurves.AddRange(hexLines);

        createtheGroup(theCurves);

    }


    //Create Group
    private NXObject createtheGroup(List<TaggedObject> theCurves)
    {
        Group nullGroup = null;

	    GroupBuilder groupBuilder1;
	    groupBuilder1 = wp.CreateGatewayGroupBuilder(nullGroup);

	    groupBuilder1.ActionType = 0;

	    groupBuilder1.ActivegroupOption = true;

	    groupBuilder1.GroupDisplayProperties = false;

	    groupBuilder1.GroupName = screwSize;

	    bool added1;
	    added1 = groupBuilder1.ObjectsInGroup.Add(theCurves.ToArray());

	    NXObject nXObject1;
	    nXObject1 = groupBuilder1.Commit();

	    groupBuilder1.Destroy();

	    dp.Views.WorkView.Regenerate();

	    return nXObject1;
    }

    //Select Line
    private TaggedObject select_line(string prompt)
    {
        UI theUI = UI.GetUI();
        TaggedObject selobj;
        Point3d cursor;
        Selection.MaskTriple[] mask = new Selection.MaskTriple[1];

        mask[0].Type = UFConstants.UF_line_type;
        mask[0].Subtype = 0;
        mask[0].SolidBodySubtype = 0;


        Selection.Response resp = theUI.SelectionManager.SelectTaggedObject(prompt, prompt,
            Selection.SelectionScope.WorkPart, Selection.SelectionAction.ClearAndEnableSpecific, false, false, mask, out selobj, out cursor);

        if (resp == Selection.Response.ObjectSelected ||
                resp == Selection.Response.ObjectSelectedByName)
        {
            return selobj;
        }
        else
            return null;

    }

    //Get selections for section view
    public bool getSelection()
    {
        TaggedObject tmpObj = null;

        tmpObj = select_line("Select Centerline of Screw");
        if (tmpObj == null)
            return false;

        centerLine = (Line)tmpObj;

        //Select Baseline for start of thread hole
        tmpObj = null;
        tmpObj = select_line("Select Baseline for Tapping");
        if (tmpObj == null)
            return false;

        baseLine = (Line)tmpObj;

        //Select Boundary line for start of counter bore/countersunk hole
        tmpObj = null;
        tmpObj = select_line("Select Boundary line for Counterbore");
        if (tmpObj == null)
            return false;
        
        boundLine = (Line)tmpObj;

        return true;

    }


    NXOpen.Point setXandYdirections()
    {
	    Point3d zeroPoint3d = new Point3d(0.0,0.0,0.0);
	    NXOpen.Point zeroPoint = wp.Points.CreatePoint(zeroPoint3d);

	    NXOpen.Point pointRef = null;
	    pointRef = wp.Points.CreatePoint(centerLine, baseLine,  zeroPoint3d, wp.Views.WorkView,
		    SmartObject.UpdateOption.WithinModeling);

	    wp.Points.RemoveParameters(pointRef);
	    pointRef.SetVisibility(SmartObject.VisibilityOption.Visible);
	    pointRef.RedisplayObject();

	    refPoint = pointRef;
	    pointRef.LineWidth = DisplayableObject.ObjectWidth.Thin;

	    NXOpen.Point pointBound = null;
	    pointBound = wp.Points.CreatePoint(centerLine, boundLine,  zeroPoint3d, wp.Views.WorkView,
		    SmartObject.UpdateOption.WithinModeling);

	    wp.Points.RemoveParameters(pointBound);
	    pointBound.SetVisibility(SmartObject.VisibilityOption.Visible);
	    pointBound.RedisplayObject();


	    double[] dir = new double[] {pointRef.Coordinates.X - pointBound.Coordinates.X,
		    pointRef.Coordinates.Y - pointBound.Coordinates.Y,
		    pointRef.Coordinates.Z - pointBound.Coordinates.Z};

	    double[] unit_dir = new double[3];
	    double magnitude;

	    ufs.Vec3.Unitize(dir, 0, out magnitude, unit_dir);

	    Vector3d yDir = new Vector3d (unit_dir[0], unit_dir[1], unit_dir[2]);

        double[] baseLineDir = {baseLine.EndPoint.X - baseLine.StartPoint.X,
						baseLine.EndPoint.Y - baseLine.StartPoint.Y,
						baseLine.EndPoint.Z - baseLine.StartPoint.Z};

	    double[] normDir = new double[3];

        ufs.Vec3.Unitize(baseLineDir, 0, out magnitude, normDir);

	    Vector3d xDir = new Vector3d(normDir[0], normDir[1], normDir[2]);

	    dir_X = xDir;
	    dir_Y = yDir;

	    return pointBound;

    }

    private Point3d offset_point(Point3d from, Vector3d dir, double dist)
    {
        double[] vec = new double[3]{ dir.X, dir.Y, dir.Z };
	    double mag;
	    ufs.Vec3.Unitize(vec, 0, out mag, vec);
	    ufs.Vec3.Scale(dist, vec, vec);

	    Point3d theOffsePoint = new Point3d(from.X + vec[0], from.Y + vec[1], from.Z + vec[2]);

        return theOffsePoint;
    }


    //Sets screw_len global variable and returns distance from base to boundary

    private void findReqdScrewLength(NXOpen.Point theBase, NXOpen.Point theBnd)
    {

        double[] baseCoords = { theBase.Coordinates.X, theBase.Coordinates.Y, theBase.Coordinates.Z };
        double[] bound = { theBnd.Coordinates.X, theBnd.Coordinates.Y, theBnd.Coordinates.Z };

        double dist_base_to_bound;
        double theLen;

        ufs.Vec3.Distance(baseCoords, bound, out dist_base_to_bound);

        base_to_bnd_dist = dist_base_to_bound;

        if (screwCategory == "Cap Screw")
            theLen = dist_base_to_bound - d_cbore_depth + d_nom_size * 1.75;
        else if (screwCategory == "Countersunk Screw")
            theLen = dist_base_to_bound - 0.5 + d_nom_size * 1.75;
        else
            theLen = dist_base_to_bound - d_cbore_depth;

        int cnt = std_screw_lens.Count;
        if (cnt == 0)
        {
            MessageBox.Show("Screw Lengths array count is zero!!  Exiting!!");
            screw_len = 0;
            return;
        }

        if (theLen >= std_screw_lens[cnt - 1])
        {
            screw_len = std_screw_lens[cnt - 1];
            return;
        }

        for (int i = 0; i < cnt - 1; i++)
        {
            if (std_screw_lens[i] < theLen && std_screw_lens[i + 1] > theLen)
            {
                screw_len = std_screw_lens[i];
                return;
            }
            else if (std_screw_lens[i] < theLen + 2 && std_screw_lens[i + 1] > theLen + 2)
            {
                screw_len = std_screw_lens[i];
                return;
            }
        }
        return;
    }


    //Draw Section view of Cap Screw
    public void drawSectionViewCapScrew()
    {
        getStdScrewLengths();

        List<TaggedObject> objsToGroup = new List<TaggedObject>();

        Session.UndoMarkId markId1;
        markId1 = s.SetUndoMark(Session.MarkVisibility.Visible, "Cap Screw");

        NXOpen.Point pointBound = null;
        pointBound = setXandYdirections();

        objsToGroup.Add(refPoint);

        ufs.Disp.SetHighlight(baseLine.Tag, 0);
        ufs.Disp.SetHighlight(boundLine.Tag, 0);
        ufs.Disp.SetHighlight(centerLine.Tag, 0);

        //get required standard screw length
        findReqdScrewLength(refPoint, pointBound);
        if (screw_len == 0)
        {
            MessageBox.Show("Something went wrong.  Screw length is zero!!");
            s.LogFile.WriteLine("Something went wrong.  Screw length is zero!!");
            s.UndoToMark(markId1, "Cap Screw");
            return;
        }

        //lw.WriteLine("Screw Length is " + screw_len.ToString());
        s.LogFile.WriteLine("Screw Length is " + screw_len.ToString());

        double d_screw_engagement = d_nom_size * 1.75;

        double reqd_cbore_depth = base_to_bnd_dist - (screw_len - d_screw_engagement);

        if (reqd_cbore_depth < d_cbore_depth)
        {
            d_screw_engagement = d_screw_engagement + d_cbore_depth - reqd_cbore_depth;
            reqd_cbore_depth = d_cbore_depth;
        }


        Point3d pointCbrHoleBnd1 = offset_point(pointBound.Coordinates, dir_X, d_cbore_dia / 2);

        Point3d pointCbrHoleBnd2 = offset_point(pointBound.Coordinates, dir_X, -d_cbore_dia / 2);

        Line cbVertLine1 = wp.Curves.CreateLine(pointCbrHoleBnd1,
            offset_point(pointCbrHoleBnd1, dir_Y, reqd_cbore_depth));
            objsToGroup.Add(cbVertLine1);

        Line cbVertLine2 = wp.Curves.CreateLine(pointCbrHoleBnd2,
            offset_point(pointCbrHoleBnd2, dir_Y, reqd_cbore_depth));
             objsToGroup.Add(cbVertLine2);

        Line cbHorizLine = wp.Curves.CreateLine(cbVertLine1.EndPoint, cbVertLine2.EndPoint);
             objsToGroup.Add(cbHorizLine);

        //Point at intersection of counter bored hole depth and CL
	    Point3d pointCbrHoleDepCL = offset_point(pointBound.Coordinates, dir_Y, reqd_cbore_depth);

	    //Point at start of clearance hole from end of counter bore hole
	    Point3d pointClrHoleStart1 = offset_point(pointCbrHoleDepCL, dir_X, d_clr_dia/2);
	    Point3d pointClrHoleStart2 = offset_point(pointCbrHoleDepCL, dir_X, -d_clr_dia/2);
	
	    Line clrVertLine1 = wp.Curves.CreateLine(pointClrHoleStart1, 
		    offset_point(refPoint.Coordinates, dir_X, d_clr_dia/2));
	    objsToGroup.Add(clrVertLine1);

	    Line clrVertLine2 = wp.Curves.CreateLine(pointClrHoleStart2, 
		    offset_point(refPoint.Coordinates, dir_X, -d_clr_dia/2));
	    objsToGroup.Add(clrVertLine2);


	    // Screw Head
	    Point3d pointScrewHead1 = offset_point(pointCbrHoleDepCL,dir_X, d_head_dia/2);
	    Point3d pointScrewHead2 = offset_point(pointCbrHoleDepCL,dir_X, -d_head_dia/2);

	    Line screwHeadVertLine1 = wp.Curves.CreateLine(pointScrewHead1, 
		    offset_point(pointScrewHead1, dir_Y, -d_nom_size * 0.9));
	    objsToGroup.Add(screwHeadVertLine1);

	    Line screwHeadVertLine2 = wp.Curves.CreateLine(pointScrewHead2, 
		    offset_point(pointScrewHead2, dir_Y, -d_nom_size * 0.9));
	    objsToGroup.Add(screwHeadVertLine2);

	    Line screwHeadHorLine1 = wp.Curves.CreateLine(screwHeadVertLine1.EndPoint, 
		    screwHeadVertLine2.EndPoint);
	    objsToGroup.Add(screwHeadHorLine1);

	    Point3d pointScrewHeadTopCL =  (offset_point(pointCbrHoleDepCL, dir_Y, -d_nom_size));

	    Point3d pointScrewHeadChamf1 = (offset_point(pointScrewHeadTopCL, dir_X, (d_head_dia/2)*0.9 ));
	    Point3d pointScrewHeadChamf2 = (offset_point(pointScrewHeadTopCL, dir_X, -(d_head_dia/2)*0.9 ));

	    Line screwHeadHorLine2 = wp.Curves.CreateLine( pointScrewHeadChamf1, pointScrewHeadChamf2 );
	    objsToGroup.Add(screwHeadHorLine2);

	    Line screwHeadChmfLine1 = wp.Curves.CreateLine( screwHeadVertLine1.EndPoint,
		    pointScrewHeadChamf1);
	    objsToGroup.Add(screwHeadChmfLine1);

	    Line screwHeadChmfLine2 = wp.Curves.CreateLine( screwHeadVertLine2.EndPoint,
		    pointScrewHeadChamf2);
	    objsToGroup.Add(screwHeadChmfLine2);

	    //Screw Body
	    Point3d pointScrewBody1 = offset_point(pointCbrHoleDepCL, dir_X, d_nom_size/2);
	    Point3d pointScrewBody2 = offset_point(pointCbrHoleDepCL, dir_X, -d_nom_size/2);

	    Point3d pointScrewBodyTipCL = (offset_point(pointCbrHoleDepCL, dir_Y, screw_len));
	    Point3d pointScrewBodyTip1 = (offset_point(pointScrewBodyTipCL, dir_X, d_tap_drill_dia/2));
	    Point3d pointScrewBodyTip2 = offset_point(pointScrewBodyTip1, dir_X, -d_tap_drill_dia);

	    Line screwBodyVertLine1 = wp.Curves.CreateLine(pointScrewBody1, 
		    offset_point(pointScrewBody1, dir_Y, screw_len - (d_nom_size/2 - d_tap_drill_dia/2)));
	    objsToGroup.Add(screwBodyVertLine1);

	    Line screwBodyVertLine2 = wp.Curves.CreateLine(pointScrewBody2, 
		    offset_point(pointScrewBody2, dir_Y, screw_len - (d_nom_size/2 - d_tap_drill_dia/2)));
	    objsToGroup.Add(screwBodyVertLine2);

	    Line screwBodyHorLine1 = wp.Curves.CreateLine(screwBodyVertLine1.EndPoint,
		    screwBodyVertLine2.EndPoint);
	    objsToGroup.Add(screwBodyHorLine1);

	    Line screwBodyHorLine2 = wp.Curves.CreateLine(pointScrewBodyTip1, pointScrewBodyTip2);
	    objsToGroup.Add(screwBodyHorLine2);

	    Line screwBodyChmfLine1 = wp.Curves.CreateLine(screwBodyVertLine1.EndPoint,
		    pointScrewBodyTip1);
	    objsToGroup.Add(screwBodyChmfLine1);

	    Line screwBodyChmfLine2 = wp.Curves.CreateLine(screwBodyVertLine2.EndPoint,
		    pointScrewBodyTip2);
	    objsToGroup.Add(screwBodyChmfLine2);

	    //Screw CL
	    Line screwCLine = wp.Curves.CreateLine(offset_point(pointBound.Coordinates, dir_Y, -3),
		    offset_point(refPoint.Coordinates, dir_Y, d_tap_drill_depth + 3));
	    screwCLine.LineFont = DisplayableObject.ObjectFont.Centerline;
	    screwCLine.LineWidth = DisplayableObject.ObjectWidth.Thin;
	    screwCLine.Color = 6;

        drawSectionViewTappedHole();

        ufs.Obj.DeleteObject(pointBound.Tag);

        int nErrs;
        nErrs = s.UpdateManager.DoUpdate(markId1);

        if (fontOption == "Hidden")
        {
            foreach (DisplayableObject theObj in objsToGroup)
            {
                setCurveFontWidthHidden(theObj);
            }
        }

        objsToGroup.Add(screwCLine);

        createtheGroup(objsToGroup);

    }

    public void drawSectionViewTappedHole()
    {
        List<TaggedObject> tapObjsToGroup = new List<TaggedObject>();

        NXOpen.Point pointBound = null;

        if (screwCategory == "Tapped Hole")
        {
            Session.UndoMarkId markId1;
            markId1 = s.SetUndoMark(Session.MarkVisibility.Visible, "Tapped Hole");

            pointBound = setXandYdirections();
        }
        // Drill point height
        double h = d_tap_drill_dia / 2 * Math.Tan(31 * Math.PI / 180.0);

        Point3d tapDrillBase1 = offset_point(refPoint.Coordinates, dir_X, d_tap_drill_dia / 2);
        Point3d tapDrillBase2 = offset_point(refPoint.Coordinates, dir_X, -d_tap_drill_dia / 2);

        Line tapDrillVertLine1 = wp.Curves.CreateLine(tapDrillBase1,
            offset_point(tapDrillBase1, dir_Y, d_tap_drill_depth - h));
        tapObjsToGroup.Add(tapDrillVertLine1);

        Line tapDrillVertLine2 = wp.Curves.CreateLine(tapDrillBase2,
            offset_point(tapDrillBase2, dir_Y, d_tap_drill_depth - h));
        tapObjsToGroup.Add(tapDrillVertLine2);

        Line tapDrillHorLine = wp.Curves.CreateLine(tapDrillVertLine1.EndPoint,
            tapDrillVertLine2.EndPoint);
        tapObjsToGroup.Add(tapDrillHorLine);

        Line tapDrillAngleLIne1 = wp.Curves.CreateLine(tapDrillVertLine1.EndPoint,
            offset_point(refPoint.Coordinates, dir_Y, d_tap_drill_depth));
        tapObjsToGroup.Add(tapDrillAngleLIne1);

        Line tapDrillAngleLIne2 = wp.Curves.CreateLine(tapDrillVertLine2.EndPoint,
            offset_point(refPoint.Coordinates, dir_Y, d_tap_drill_depth));
        tapObjsToGroup.Add(tapDrillAngleLIne2);

        Point3d tapBase1 = offset_point(refPoint.Coordinates, dir_X, d_tap_size / 2);
        Point3d tapBase2 = offset_point(refPoint.Coordinates, dir_X, -d_tap_size / 2);

        Line tapVertLine1 = wp.Curves.CreateLine(tapBase1, offset_point(tapBase1, dir_Y, d_tap_depth));
        tapObjsToGroup.Add(tapVertLine1);

        Line tapVertLine2 = wp.Curves.CreateLine(tapBase2, offset_point(tapBase2, dir_Y, d_tap_depth));
        tapObjsToGroup.Add(tapVertLine2);

        Line tapSizeHorLine = wp.Curves.CreateLine(tapVertLine1.EndPoint,
            tapVertLine2.EndPoint);
        tapObjsToGroup.Add(tapSizeHorLine);

        Line tapAngleLine1 = wp.Curves.CreateLine(tapVertLine1.EndPoint,
            offset_point(tapDrillBase1, dir_Y, d_tap_depth + (d_tap_size / 2 - d_tap_drill_dia / 2)));
        tapObjsToGroup.Add(tapAngleLine1);

        Line tapAngleLine2 = wp.Curves.CreateLine(tapVertLine2.EndPoint,
            offset_point(tapDrillBase2, dir_Y, d_tap_depth + (d_tap_size / 2 - d_tap_drill_dia / 2)));
        tapObjsToGroup.Add(tapAngleLine2);

        Line tapAngleHorLine = wp.Curves.CreateLine(tapAngleLine1.EndPoint,
            tapAngleLine2.EndPoint);
        tapObjsToGroup.Add(tapAngleHorLine);

        if (fontOption == "Hidden")
        {
            foreach (DisplayableObject theObj in tapObjsToGroup)
            {
                setCurveFontWidthHidden(theObj);
            }
        }

        if (screwCategory == "Tapped Hole")
        {
            Line theCLine = wp.Curves.CreateLine(offset_point(refPoint.Coordinates, dir_Y, -3),
                offset_point(tapDrillAngleLIne1.EndPoint, dir_Y, 3));
            theCLine.LineFont = DisplayableObject.ObjectFont.Centerline;
            theCLine.LineWidth = DisplayableObject.ObjectWidth.Thin;
            theCLine.Color = 6;

            tapObjsToGroup.Add(theCLine);

            tapObjsToGroup.Add(refPoint);

            ufs.Obj.DeleteObject(pointBound.Tag);
        }

        createtheGroup(tapObjsToGroup);

    }


    //Draw Section View CSK Screw
    public void drawSectionViewCSKScrew()
    {
        getStdScrewLengths();

        List<TaggedObject> objsToGroup = new List<TaggedObject>();

        Session.UndoMarkId markId1;
        markId1 = s.SetUndoMark(Session.MarkVisibility.Visible, "CSK Screw");

        NXOpen.Point pointBound = null;

        pointBound = setXandYdirections();

        objsToGroup.Add(refPoint);

        //get required screw length
        findReqdScrewLength(refPoint, pointBound);
        if (screw_len == 0)
        {
            MessageBox.Show("Something went wrong.  Screw length is zero!!");
            s.LogFile.WriteLine("Something went wrong.  Screw length is zero!!");
            s.UndoToMark(markId1, "CSK Screw");
            return;
        }

        //lw.WriteLine("Screw Length is " + screw_len.ToString());
        s.LogFile.WriteLine("Screw Length is " + screw_len.ToString());

        double csk_head_depth = (Math.Tan((90 - d_csk_head_angle / 2) * Math.PI / 180.0) * (d_csk_dia / 2 - d_head_dia / 2)) - d_csk_head_flat;

        double csk_depth = (Math.Tan((90 - d_csk_head_angle / 2) * Math.PI / 180.0) * (d_csk_dia / 2 - d_clr_dia / 2));

        double d_screw_engagement = screw_len + csk_head_depth - base_to_bnd_dist;

        if (d_screw_engagement < 1.5 * d_nom_size)
        {
            string msg = "Countersunk Screw thread engagement is " + d_screw_engagement.ToString() + ", which is less than 1.5 times.  Do you wish to continue?";
            int resp = theUI.NXMessageBox.Show("Countersunk Screw", NXMessageBox.DialogType.Question, msg);
            if (resp == 2)
            {
                s.UndoToMark(markId1, "CSK Screw");
                return;
            }
        }



        Point3d pointCskHoleBnd1 = offset_point(pointBound.Coordinates, dir_X, d_csk_dia / 2);

        Point3d pointCskHoleBnd2 = offset_point(pointBound.Coordinates, dir_X, -d_csk_dia / 2);

        //Point at intersection of countersunk hole depth and CL
        Point3d pointCskDepthCL = offset_point(pointBound.Coordinates, dir_Y, csk_depth);

        //Point at start of clearance hole from end of counter bore hole
        Point3d pointClrHoleStart1 = offset_point(pointCskDepthCL, dir_X, d_clr_dia / 2);
        Point3d pointClrHoleStart2 = offset_point(pointCskDepthCL, dir_X, -d_clr_dia / 2);

        Line cskAngleLine1 = wp.Curves.CreateLine(pointCskHoleBnd1, pointClrHoleStart1);
        objsToGroup.Add(cskAngleLine1);

        Line cskAngleLine2 = wp.Curves.CreateLine(pointCskHoleBnd2, pointClrHoleStart2);
        objsToGroup.Add(cskAngleLine2);

        Line clrVertLine1 = wp.Curves.CreateLine(pointClrHoleStart1,
            offset_point(refPoint.Coordinates, dir_X, d_clr_dia / 2));
        objsToGroup.Add(clrVertLine1);

        Line clrVertLine2 = wp.Curves.CreateLine(pointClrHoleStart2,
            offset_point(refPoint.Coordinates, dir_X, -d_clr_dia / 2));
        objsToGroup.Add(clrVertLine2);

        drawSectionViewTappedHole();

        // Draw Countersunk Screw
        // Screw Head

        Point3d pointScrewHeadStartCL = offset_point(pointBound.Coordinates, dir_Y, csk_head_depth);

        Point3d pointScrewHead1 = offset_point(pointScrewHeadStartCL, dir_X, d_head_dia / 2);
        Point3d pointScrewHead2 = offset_point(pointScrewHeadStartCL, dir_X, -d_head_dia / 2);

        Line screwHeadVertLine1 = wp.Curves.CreateLine(pointScrewHead1,
            offset_point(pointScrewHead1, dir_Y, d_csk_head_flat));
        objsToGroup.Add(screwHeadVertLine1);

        Line screwHeadVertLine2 = wp.Curves.CreateLine(pointScrewHead2,
            offset_point(pointScrewHead2, dir_Y, d_csk_head_flat));
        objsToGroup.Add(screwHeadVertLine2);

        Line screwHeadHorLine1 = wp.Curves.CreateLine(screwHeadVertLine1.StartPoint,
            screwHeadVertLine2.StartPoint);
        objsToGroup.Add(screwHeadHorLine1);

        Line screwHeadHorLine2 = wp.Curves.CreateLine(screwHeadVertLine1.EndPoint,
            screwHeadVertLine2.EndPoint);
        objsToGroup.Add(screwHeadHorLine2);

        //Intersection of CSK flat line and CL (Mid point of 'screwHeadHorLine2'
        Point3d pointMidHeadHorLine2 = offset_point(pointScrewHeadStartCL, dir_Y, d_csk_head_flat);

        //Head height from flat
        double head_ht_from_flat_to_body = (Math.Tan((90 - d_csk_head_angle / 2) * Math.PI / 180.0) * (d_head_dia / 2 - d_nom_size / 2));

        //Intersection of CSK angle and Body CL
        Point3d pointScrewHeadTopCL = offset_point(pointMidHeadHorLine2, dir_Y, head_ht_from_flat_to_body);

        //Screw Body
        Point3d pointScrewBody1 = offset_point(pointScrewHeadTopCL, dir_X, d_nom_size / 2);
        Point3d pointScrewBody2 = offset_point(pointScrewHeadTopCL, dir_X, -d_nom_size / 2);

        Point3d pointScrewBodyTipCL = offset_point(pointScrewHeadStartCL, dir_Y, screw_len);
        Point3d pointScrewBodyTip1 = offset_point(pointScrewBodyTipCL, dir_X, d_tap_drill_dia / 2);
        Point3d pointScrewBodyTip2 = offset_point(pointScrewBodyTip1, dir_X, -d_tap_drill_dia);

        Line screwHeadAngleLine1 = wp.Curves.CreateLine(screwHeadVertLine1.EndPoint, pointScrewBody1);
        objsToGroup.Add(screwHeadAngleLine1);

        Line screwHeadAngleLine2 = wp.Curves.CreateLine(screwHeadVertLine2.EndPoint, pointScrewBody2);
        objsToGroup.Add(screwHeadAngleLine2);

        Line screwBodyVertLine1 = wp.Curves.CreateLine(pointScrewBody1,
            offset_point(pointScrewBody1, dir_Y, screw_len - (d_nom_size / 2 - d_tap_drill_dia / 2) - head_ht_from_flat_to_body - d_csk_head_flat));
        objsToGroup.Add(screwBodyVertLine1);

        Line screwBodyVertLine2 = wp.Curves.CreateLine(pointScrewBody2,
            offset_point(pointScrewBody2, dir_Y, screw_len - (d_nom_size / 2 - d_tap_drill_dia / 2) - head_ht_from_flat_to_body - d_csk_head_flat));
        objsToGroup.Add(screwBodyVertLine2);

        Line screwBodyHorLine1 = wp.Curves.CreateLine(screwBodyVertLine1.EndPoint,
            screwBodyVertLine2.EndPoint);
        objsToGroup.Add(screwBodyHorLine1);

        Line screwBodyHorLine2 = wp.Curves.CreateLine(pointScrewBodyTip1, pointScrewBodyTip2);
        objsToGroup.Add(screwBodyHorLine2);

        Line screwBodyHorLine3 = wp.Curves.CreateLine(pointScrewBody1, pointScrewBody2);
        objsToGroup.Add(screwBodyHorLine3);

        Line screwBodyChmfLine1 = wp.Curves.CreateLine(screwBodyVertLine1.EndPoint,
            pointScrewBodyTip1);
        objsToGroup.Add(screwBodyChmfLine1);

        Line screwBodyChmfLine2 = wp.Curves.CreateLine(screwBodyVertLine2.EndPoint,
            pointScrewBodyTip2);
        objsToGroup.Add(screwBodyChmfLine2);

        //Screw CL
        Line screwCLine = wp.Curves.CreateLine(offset_point(pointBound.Coordinates, dir_Y, -3),
            offset_point(refPoint.Coordinates, dir_Y, d_tap_drill_depth + 3));
        screwCLine.LineFont = DisplayableObject.ObjectFont.Centerline;
        screwCLine.LineWidth = DisplayableObject.ObjectWidth.Thin;
        screwCLine.Color = 6;

        ufs.Obj.DeleteObject(pointBound.Tag);

        int nErrs2;
        nErrs2 = s.UpdateManager.DoUpdate(markId1);

        if (fontOption == "Hidden")
        {
            foreach (DisplayableObject theObj in objsToGroup)
            {
                setCurveFontWidthHidden(theObj);
            }
        }

        objsToGroup.Add(screwCLine);

        createtheGroup(objsToGroup);

    }


    public void drawSectionViewShoulderScrew()
    {
        getStdScrewLengths();

        List<TaggedObject> objsToGroup = new List<TaggedObject>();

        Session.UndoMarkId markId1;
        markId1 = s.SetUndoMark(Session.MarkVisibility.Visible, "ShoulderScrew");

        NXOpen.Point pointBound = null;

        pointBound = setXandYdirections();

        objsToGroup.Add(refPoint);

        //get required screw length
        findReqdScrewLength(refPoint, pointBound);
        if (screw_len == 0)
        {
            MessageBox.Show("Something went wrong.  Screw length is zero!!");
            s.LogFile.WriteLine("Something went wrong.  Screw length is zero!!");
            s.UndoToMark(markId1, "ShoulderScrew");
            return;
        }

        //lw.WriteLine("Screw Length is " + screw_len.ToString());
        s.LogFile.WriteLine("Screw Length is " + screw_len.ToString());

        double d_screw_engagement = d_thread_len;

        double reqd_cbore_depth = base_to_bnd_dist - screw_len + d_shoulder_screw_depth_clearance;

        Point3d pointCbrHoleBnd1 = offset_point(pointBound.Coordinates, dir_X, d_cbore_dia / 2);

        Point3d pointCbrHoleBnd2 = offset_point(pointBound.Coordinates, dir_X, -d_cbore_dia / 2);

        Line cbVertLine1 = wp.Curves.CreateLine(pointCbrHoleBnd1,
             offset_point(pointCbrHoleBnd1, dir_Y, reqd_cbore_depth));
        objsToGroup.Add(cbVertLine1);

        Line cbVertLine2 = wp.Curves.CreateLine(pointCbrHoleBnd2,
            offset_point(pointCbrHoleBnd2, dir_Y, reqd_cbore_depth));
        objsToGroup.Add(cbVertLine2);

       // Line cbHorizLine = wp.Curves.CreateLine(cbVertLine1.EndPoint, cbVertLine2.EndPoint);
        Line cbHorizLine1 = wp.Curves.CreateLine(cbVertLine1.EndPoint, 
            offset_point(cbVertLine1.EndPoint, dir_X, d_nom_size/2 - d_cbore_dia/2));
        objsToGroup.Add(cbHorizLine1);

        Line cbHorizLine2 = wp.Curves.CreateLine(cbVertLine2.EndPoint,
            offset_point(cbVertLine2.EndPoint, dir_X, d_cbore_dia/2 - d_nom_size/2));
        objsToGroup.Add(cbHorizLine2);

        //Point at intersection of counter bored hole depth and CL
        Point3d pointCbrHoleDepCL = offset_point(pointBound.Coordinates, dir_Y, reqd_cbore_depth);

        //Point at start of clearance hole from end of counter bore hole
        Point3d pointClrHoleStart1 = offset_point(pointCbrHoleDepCL, dir_X, d_clr_dia / 2);
        Point3d pointClrHoleStart2 = offset_point(pointCbrHoleDepCL, dir_X, -d_clr_dia / 2);

        Line clrVertLine1 = wp.Curves.CreateLine(pointClrHoleStart1,
            offset_point(refPoint.Coordinates, dir_X, d_clr_dia / 2));
        objsToGroup.Add(clrVertLine1);

        Line clrVertLine2 = wp.Curves.CreateLine(pointClrHoleStart2,
            offset_point(refPoint.Coordinates, dir_X, -d_clr_dia / 2));
        objsToGroup.Add(clrVertLine2);

        drawSectionViewTappedHole();

        // Draw Shoulder Screw
        // Screw Head

        //Point at intersection of counter bored hole depth and CL
        Point3d pointScrewHeadCL = offset_point(pointBound.Coordinates, dir_Y, reqd_cbore_depth-d_shoulder_screw_depth_clearance);

        Point3d pointScrewHead1 = offset_point(pointScrewHeadCL, dir_X, d_head_dia / 2);
        Point3d pointScrewHead2 = offset_point(pointScrewHeadCL, dir_X, -d_head_dia / 2);

        Line screwHeadVertLine1 = wp.Curves.CreateLine(pointScrewHead1,
            offset_point(pointScrewHead1, dir_Y, -d_nom_size * 0.9));
        objsToGroup.Add(screwHeadVertLine1);

        Line screwHeadVertLine2 = wp.Curves.CreateLine(pointScrewHead2,
            offset_point(pointScrewHead2, dir_Y, -d_nom_size * 0.9));
        objsToGroup.Add(screwHeadVertLine2);

        Line screwHeadHorLine0 = wp.Curves.CreateLine(screwHeadVertLine1.StartPoint,
            screwHeadVertLine2.StartPoint);
        objsToGroup.Add(screwHeadHorLine0);

        Line screwHeadHorLine1 = wp.Curves.CreateLine(screwHeadVertLine1.EndPoint,
            screwHeadVertLine2.EndPoint);
        objsToGroup.Add(screwHeadHorLine1);

        Point3d pointScrewHeadTopCL = offset_point(pointScrewHeadCL, dir_Y, -d_nom_size);

        Point3d pointScrewHeadChamf1 = offset_point(pointScrewHeadTopCL, dir_X, (d_head_dia / 2) * 0.9);
        Point3d pointScrewHeadChamf2 = offset_point(pointScrewHeadTopCL, dir_X, -(d_head_dia / 2) * 0.9);

        Line screwHeadHorLine2 = wp.Curves.CreateLine(pointScrewHeadChamf1, pointScrewHeadChamf2);
        objsToGroup.Add(screwHeadHorLine2);

        Line screwHeadChmfLine1 = wp.Curves.CreateLine(screwHeadVertLine1.EndPoint,
            pointScrewHeadChamf1);
        objsToGroup.Add(screwHeadChmfLine1);

        Line screwHeadChmfLine2 = wp.Curves.CreateLine(screwHeadVertLine2.EndPoint,
            pointScrewHeadChamf2);
        objsToGroup.Add(screwHeadChmfLine2);

        //Shoulder Screw Body
        Point3d pointScrewBody1 = offset_point(pointScrewHeadCL, dir_X, d_nom_size / 2);
        Point3d pointScrewBody2 = offset_point(pointScrewHeadCL, dir_X, -d_nom_size / 2);

        Point3d pointScrewBodyTipCL = offset_point(pointScrewHeadCL, dir_Y, screw_len + d_thread_len);
        Point3d pointScrewBodyTip1 = offset_point(pointScrewBodyTipCL, dir_X, d_tap_drill_dia / 2);
        Point3d pointScrewBodyTip2 = offset_point(pointScrewBodyTip1, dir_X, -d_tap_drill_dia);

        Line screwBodyVertLine1 = wp.Curves.CreateLine(pointScrewBody1,
            offset_point(pointScrewBody1, dir_Y, screw_len));
        objsToGroup.Add(screwBodyVertLine1);

        Line screwBodyVertLine2 = wp.Curves.CreateLine(pointScrewBody2,
            offset_point(pointScrewBody2, dir_Y, screw_len));
        objsToGroup.Add(screwBodyVertLine2);

        Line screwBodyHorLine1 = wp.Curves.CreateLine(screwBodyVertLine1.EndPoint,
            screwBodyVertLine2.EndPoint);
        objsToGroup.Add(screwBodyHorLine1);

        Line screwBodyHorLine2 = wp.Curves.CreateLine(pointScrewBodyTip1, pointScrewBodyTip2);
        objsToGroup.Add(screwBodyHorLine2);

        Point3d pointScrewThreadStart1 = offset_point(refPoint.Coordinates, dir_X, d_tap_size / 2);
        Point3d pointScrewThreadStart2 = offset_point(refPoint.Coordinates, dir_X, -d_tap_size / 2);

        Line screwThreadLine1 = wp.Curves.CreateLine(pointScrewThreadStart1,
            offset_point(pointScrewThreadStart1, dir_Y, d_thread_len - (d_tap_size / 2 - d_tap_drill_dia / 2)));
        objsToGroup.Add(screwThreadLine1);

        Line screwThreadLine2 = wp.Curves.CreateLine(pointScrewThreadStart2,
            offset_point(pointScrewThreadStart2, dir_Y, d_thread_len - (d_tap_size / 2 - d_tap_drill_dia / 2)));
        objsToGroup.Add(screwThreadLine2);

        Line screwBodyChmfLine1 = wp.Curves.CreateLine(screwThreadLine1.EndPoint,
            pointScrewBodyTip1);
        objsToGroup.Add(screwBodyChmfLine1);

        Line screwBodyChmfLine2 = wp.Curves.CreateLine(screwThreadLine2.EndPoint,
            pointScrewBodyTip2);
        objsToGroup.Add(screwBodyChmfLine2);

        Line screwThreadHorLine1 = wp.Curves.CreateLine(screwThreadLine1.EndPoint,
            screwThreadLine2.EndPoint);
        objsToGroup.Add(screwThreadHorLine1);

        //Screw CL
        Line screwCLine = wp.Curves.CreateLine(offset_point(pointBound.Coordinates, dir_Y, -3),
            offset_point(refPoint.Coordinates, dir_Y, d_tap_drill_depth + 3));
        screwCLine.LineFont = DisplayableObject.ObjectFont.Centerline;
        screwCLine.LineWidth = DisplayableObject.ObjectWidth.Thin;
        screwCLine.Color = 6;

        ufs.Obj.DeleteObject(pointBound.Tag);

        int nErrs2;
        nErrs2 = s.UpdateManager.DoUpdate(markId1);

        if (fontOption == "Hidden")
        {
            foreach (DisplayableObject theObj in objsToGroup)
            {
                setCurveFontWidthHidden(theObj);
            }
        }

        objsToGroup.Add(screwCLine);

        createtheGroup(objsToGroup);

    }
}